/*
    This is a demonstration of the Marcotte 3-D Wave Algorithm. - Gabriel La Frenire, June 10, 2011.
    This program in C language was edited in Code::Blocks: http://www.codeblocks.org/downloads/binaries
    and compiled by means of MinGW (GNU GCC).
    The SDL library for graphics is available at: www.libsdl.org
    The SDL_ttf library for printing text: http://www.libsdl.org/projects/SDL_ttf/
    Please indicate the SDL_ttf library path in the build options, for example (Windows):
        C:\Program Files (x86)\CodeBlocks\MinGW\lib\SDL_ttf.lib
    The following files must be present in the program directory:
        DoulosSILR.ttf
        libfreetype-6.dll
        SDL.dll
        SDL_ttf.dll
        zlib1.dll
    This program runs faster using the compiler setting "Optimize even more (for speed) [-O2]".
*/

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <SDL.h>
#include <SDL_ttf.h>

#define WIDTH (624) // Fit screen to available memory.
#define HEIGHT (624) // most video formats require multiples of 2-4 pixels.
#define DEPTH (624)

void omniDirectionalEmitter(int lambda, int image);
void reverseDirection();
char sdlEvent();
void setPixel(SDL_Surface *surface, int x, int y, Uint32 pixel);
void waveAlgorithm();

float array1 [WIDTH][HEIGHT][DEPTH]; // global 3-D arrays of variables for past, present and trend.
float array2 [WIDTH][HEIGHT][DEPTH];

float (*past)[HEIGHT][DEPTH]; // pointers for past, present and trend (using only two arrays).
float (*present)[HEIGHT][DEPTH];
float (*trend)[HEIGHT][DEPTH];

const float pi = 4 * atan(1);

int main (int argc, char** argv) // MAIN ###########################################################
{
    TTF_Init();
    SDL_Init(SDL_INIT_VIDEO); // initialize SDL video.
    atexit(SDL_Quit);         // make sure SDL cleans up before exit.
    SDL_Surface *screen = NULL, *text[6] = {NULL}, *background = NULL, *rectangle[1] = {NULL}; // create a new window.
    screen = SDL_SetVideoMode(WIDTH, HEIGHT, 24, SDL_HWSURFACE);                               // 1280 x 720 24 bits resolution.
    SDL_Rect position;
    rectangle[0] = SDL_CreateRGBSurface(SDL_HWSURFACE, 1, 20, 24, 0, 0, 0, 0); // surface allocation for 2 identical rectangles.
    SDL_WM_SetCaption("3-D Spherical Standing Waves.  624 pixel cubic medium.  June 10, 2011", NULL);
    past = array1; // the goal is to obtain past = present and present = trend using Mr. Marcotte's wave algorithm.
    present = array2;
    trend = array1;

    int x, y, z, r, g, b; // Cartesian coordinates and RGB color variables.
    int previous, amplitude, gap, yCoord;
    int image = 0, lambda = 48; // wavelength (multiple of 4 required).
    float radian, curve, phase, impulse;
    bool done = false, bitmap = false; // set bitmap = true for bitmap sequence (caution: produces large number of .bmp files).
    bool emit = true;
    char message, imageNo[5];

    for (x = 1; x < WIDTH; x++) { // non-zero initialization.
        for (y = 1; y < HEIGHT; y++) {
            for (z = 1; z < DEPTH; z++) {
                array1[x][y][z] = .00001;
                array2[x][y][z] = .00001;
    }}}

    SDL_Color white = {255, 255, 255};
    TTF_Font *font_16 = NULL;
    TTF_Font *font_18 = NULL;
    font_16 = TTF_OpenFont("DoulosSILR.ttf", 16 );
    font_18 = TTF_OpenFont("DoulosSILR.ttf", 18 );
    text[1] = TTF_RenderText_Blended(font_18, "The Marcotte Three-Dimensional Wave Medium.", white);
    text[2] = TTF_RenderText_Blended(font_16, "Creating spherical standing waves.", white);
    text[3] = TTF_RenderText_Blended(font_16, "To save a screen capture, press B.", white);
    text[4] = TTF_RenderText_Blended(font_16, "Blue curve: amplitude. Red curve: impulse.", white);
    text[5] = TTF_RenderText_Blended(font_16, "www.glafreniere.com", white);

    while (!done) // PROGRAM MAIN LOOP #############################################################
    {
        // DRAWING STARTS HERE.
        SDL_FillRect(screen, 0, SDL_MapRGB(screen->format, 0, 0, 0)); // clear screen.
        phase = 2 * pi * image / lambda; // amplitude modulation.

        for (x = 0; x < WIDTH; x++) // drawing waves which are present at the center of the z coordinates only.
        {
            for (y = 0; y < HEIGHT-1; y++)
            {
                amplitude = present[x][y][DEPTH / 2]; // color distribution (center of the z plane).
                b = abs(.5 * amplitude);   // produces complementary magenta and emerald green.
                if (amplitude > 0)
                {
                    g = amplitude; // green color for positive amplitude.
                    if (g > 255) r = g - 255; else r = 0;
                }
                else
                {
                    r = -amplitude; // red color for negative amplitude.
                    if (r > 255) g = r - 255; else g = 0;
                }
                if (r > 255) r = 255;
                if (g > 255) g = 255;
                if (b > 255) b = 255;
                setPixel(screen, x, y, SDL_MapRGB(screen->format, r, g, b)); // draw pixel.
            }
            setPixel(screen, x, HEIGHT/2 , SDL_MapRGB(screen->format, 0,0,0)); // central axis.
            amplitude = .037 * present[x][HEIGHT/2][DEPTH / 2];
            if (x == 0) previous = amplitude;
            setPixel(screen, x, HEIGHT/2 - amplitude, SDL_MapRGB(screen->format, 0, 128, 255)); // amplitude curve.
            gap = amplitude - previous; // occasional empty spaces (gap > 1) to fill up.
            if (gap > 1) {
                for (yCoord = (previous + 1); yCoord < amplitude; yCoord++) {
                    setPixel(screen, x, HEIGHT/2 - yCoord , SDL_MapRGB(screen->format, 0, 128, 255));
                }
            }
            else if (gap < -1) {
                for (yCoord = (previous - 1); yCoord > amplitude; yCoord--) {
                    setPixel(screen, x, HEIGHT/2 - yCoord , SDL_MapRGB(screen->format, 0, 128, 255));
                }
            }
            previous = amplitude;
            radian = 2 * pi * abs(WIDTH / 2 - x) / lambda;
            if (emit && radian < pi) // draw impulse.
            {
                if (radian < .001) radian = .001; // avoid division by zero.
                curve = sin(radian) / radian; // sinus cardinalis (Jocelyn Marcotte's electron curve).
                yCoord = HEIGHT/2 - int(60 * cos(phase) * curve);
                setPixel(screen, x, yCoord, SDL_MapRGB(screen->format, 255, 0, 0)); // draw impulse.
                setPixel(screen, x, yCoord+1, SDL_MapRGB(screen->format, 255, 0, 0)); // draw impulse.
                setPixel(screen, x, yCoord-1, SDL_MapRGB(screen->format, 255, 0, 0)); // draw impulse.
            }
        }

        position.x = WIDTH/2-lambda/2; // full lambda central core limits.
        position.y = HEIGHT/2 - 10;
        SDL_FillRect(rectangle[0], NULL, SDL_MapRGB(screen->format, 128, 128, 128));
        SDL_BlitSurface(rectangle[0], NULL, screen, &position);
        position.x += lambda;
        SDL_FillRect(rectangle[0], NULL, SDL_MapRGB(screen->format, 128, 128, 128));
        SDL_BlitSurface(rectangle[0], NULL, screen, &position);
        if (image < .5 * WIDTH)
        {
            position.x = .5 * WIDTH + image; // moving cursor to check the system speed and also the wave speed.
            position.y = .5 * HEIGHT - 10;
            SDL_FillRect(rectangle[0], NULL, SDL_MapRGB(screen->format, 255, 255, 255));
            SDL_BlitSurface(rectangle[0], NULL, screen, &position);
        }
        position.x = 10;
        position.y = 0;
        SDL_BlitSurface(text[1], NULL, screen, &position); // Blit text.
        position.x = 10;
        position.y = 20;
        SDL_BlitSurface(text[2], NULL, screen, &position); // Blit text.
        if (!bitmap) {
            position.x = 10;
            position.y = 580;
            SDL_BlitSurface(text[3], NULL, screen, &position);
        }
        position.x = 10;
        position.y = 600;
        SDL_BlitSurface(text[4], NULL, screen, &position);
        position.x = 470;
        position.y = 600;
        SDL_BlitSurface(text[5], NULL, screen, &position);

        if (bitmap) // warning: "bitmap = true" creates a huge bitmap sequence.
        {
            if (image < 10) { // format/display image number.
                sprintf(imageNo, "capture000%d.bmp", image); }
            else if (image < 100) {
                sprintf(imageNo, "capture00%d.bmp", image); }
            else if (image < 1000) {
                sprintf(imageNo, "capture0%d.bmp", image); }
            else {
                sprintf(imageNo, "capture%d.bmp", image);
            }
            text[6] = TTF_RenderText_Blended(font_16, imageNo, white);
            position.x = 10;
            position.y = 580;
            SDL_BlitSurface(text[6], NULL, screen, &position); // blit bitmap number if required.
            SDL_SaveBMP(screen, imageNo); // save bitmap image to disk in the current folder.
            if (image == 999) done = true;
        }
        SDL_Flip(screen); // finally, update the screen.
        // DRAWING ENDS HERE.

        if (emit) // emission duration may somewhat exceed screen width (damping zone impracticable in 3-D).
        {
            omniDirectionalEmitter(lambda, image);
            if (image == 10 * lambda) emit = false; // stop emitting.
        }

        waveAlgorithm(); // using the Delmotte-Marcotte virtual medium.
        message = sdlEvent(); // keyboard and mouse management.
        if (message == 'B') SDL_SaveBMP(screen, "screen_capture.bmp");
        if (message == 'X') done = true;
        if (message == 'R' && image > 11 * lambda) reverseDirection();
        if (image < 5000) image ++; // the wave speed is 1 pixel per loop so that the system needs lambda * images to produce a full wavelength.
        if (image == 12 * lambda) reverseDirection(); // the goal is to obtain spherical standing waves (the electron prototype).

    }           // END MAIN LOOP ###################################################################
    return (0);
}               // END MAIN ########################################################################



void omniDirectionalEmitter(int lambda, int image) // generating 3-D spherical waves.
{
    int x, y, z; // three-dimensional Cartesian coordinates.
    float distance, curve, radian, impulse = 1000000 / (lambda * lambda * lambda);
    float xSquared, ySquared, zSquared; // float advisable for use with sqrt operator below.
    int halfLambda = lambda / 2; // full lambda cubic area below.
    float phase = 2 * pi * image / lambda; // amplitude modulation.

    for (x = -halfLambda; x < halfLambda; x++) {
        xSquared = x * x;
        for (y = -halfLambda; y < halfLambda; y++) {
            ySquared = y * y;
            for (z = -halfLambda; z < halfLambda; z++) {
                zSquared = z * z; // scanning the cube in-depth.
                distance = sqrt(xSquared + ySquared + zSquared); // distance to the center (Pythagoras).
                radian = 2 * pi * distance / lambda;
                if (distance <= halfLambda) { // spherical source (radius = lambda / 2).
                    if (radian < .001) radian = .001; // avoid division by zero.
                    curve = sin(radian) / radian; // sinus cardinalis (Jocelyn Marcotte's electron curve).
                    present[x+WIDTH/2][y+HEIGHT/2][z+DEPTH/2] += cos(phase) * curve * impulse;
                    trend[x+WIDTH/2][y+HEIGHT/2][z+DEPTH/2] -= cos(phase) * curve * impulse;
                }
    }}}
}

void reverseDirection() // Press R to reverse the wave direction.
{
    past = present; // a premilinary array rotation must be performed.
    present = trend;
    trend = past;// equivalent to: trend[x][y][z] = past[x][y][z] in a 3-D loop.
}

char sdlEvent()
{
    SDL_Event event;
    while (SDL_PollEvent(&event)) // message processing loop.
    {
        switch (event.type) // check for messages.
        {
        case SDL_QUIT: // exit if the window is closed.
            return 'X';
            break;
        case SDL_KEYDOWN: // check for keypresses.
            switch( event.key.keysym.sym )
            {
            case SDLK_ESCAPE: // exit if ESCAPE is pressed.
                return 'X';
                break;
            case SDLK_r: // R/r is pressed: reverse direction.
                return 'R';
                break;
            case SDLK_b: // B/b is pressed: screen capture.
                return 'B';
                break;
            }
            break;
        }
    }
    return '\0';
}

void setPixel(SDL_Surface *surface, int x, int y, Uint32 pixel)
{
//  int bpp = surface->format->BytesPerPixel;
    Uint8 *p = (Uint8 *)surface->pixels + y * surface->pitch + x * 3; // bpp = 3
    *(Uint32 *)p = pixel;
}


void waveAlgorithm()
{
    int x, y, z; // three-dimensional Cartesian coordinates.
    float orthogonal, diagonal, vertices; // influence from neighboring cells.

/*
Normally, the first part of Mr. Marcotte's 3-D algorithm uses all three x, y, z coordinates:
    for (x = 0; x < WIDTH; x++) { // updating previous states.
        for (y = 0; y < HEIGHT; y++) {
            for (z = 0; z < DEPTH; z++) {
                past   [x][y][z] = present[x][y][z]; // "THE PAST IS A GUIDE TO THE FUTURE".
                present[x][y][z] = trend  [x][y][z];
    }}}

However, rotating arrays using pointers is much faster:
    past = present;
    present = trend;
    trend = past;

The second part concerns Mr. Marcotte's enhanced 3-D trend extrapolation. It produces a
highly accurate wave propagation, considering that the influence of more distant cells
must be weaker in accord with the inverse square of the distance law. Considering one
cell at a time, the small cubic area around it contains 3 * 3 * 3 - 1 = 26 neighboring
cells which are to be separated into three groups: orthogonal, diagonal, vertices.

Relative individual influence:
    orthogonal                    1 / 1 (nominal, as a reference).
    diagonal    (1 / sqrt(2))^2 = 1 / 2
    vertices    (1 / sqrt(3))^2 = 1 / 3

Number of cells in a 3x3x3 cube exerting some influence on the central one:
    orthogonal  6
    diagonal  12
    vertices   8
    total      26 cells.

Relative influence:
    orthogonal  6 / 26 / 1 = .23076923    (3 / 13)
    diagonal  12 / 26 / 2 = .23076923    (3 / 13)  (orthogonal and diagonal are equal).
    vertices   8 / 26 / 3 = .1025641     (4 / 39)

Normalized influence:
    orthogonal      nominal, 6 cells:       (3 / 13)          = .23076923    gain = 3 / 13
    diagonal       12 normalized to 6:     (3 / 13) * 6 / 12 = .1153846     gain = 3 / 26
    vertices        8 normalized to 6:     (4 / 39) * 6 / 8  = .076923      gain = 1 / 13

Symmetrical reactance of the central cell in order to maintain perfect energy equilibrium:
    past[x][y][z]       reaction = -1 (constant).
    present[x][y][z]    reaction = X (see equation below: all versions of Marcotte's algorithm must balance to unity).

The goal is to obtain:
    6 * .23076923 + 12 * .1153846 + 8 * .076923  + X - 1 = 1
    6 * (3 / 13)  + 12 * (3 / 26) + 8 * (1 / 13) + X - 1 = 1
       (18 / 13)  +     (18 / 13) +     (8 / 13) + X - 1 = 1  (orthogonal and diagonal are still equal).
                                       (44 / 13) + X - 1 = 1
Hence:
    X = 1 + 1 - (44 / 13)
    X = -1.384615 = -18 / 13 (negative) applies to present[x][y][z] below. The wave speed is 1 pixel per loop.

On a string (one dimension only), Marcotte's algorithm simplifies to:
    for (x = 0; x < WIDTH; x++) {
        past[x] = present[x];
        present[x] = trend[x];
    }
    for (x = 1; x < WIDTH-1; x++) {
        trend[x] = present[x-1] + present[x+1] - past[x]; // here again: 1 + 1 - 1 = 1 (wave speed = 1 pixel)
    }
This procedure may be linked to Euler's method.
It can produce sine and/or cosine values using 2 * pi multiples (or 360 pixels for degrees) as wavelength.

******************* MR. MARCOTTE'S ALGORITHM IS DEFINITELY THE SIMPLEST POSSIBLE *******************

In 2-D, the wave speed is also 1 pixel per loop using the following algorithm, which is the best one:
    for (x = 0; x < WIDTH; x++) {
        for (y = 0; y < HEIGHT; y++) {
            past[x][y] = present[x][y];
            present[x][y] = trend[x][y];
    }}
    for (x = 1; x < WIDTH-1; x++) { // trend extrapolation using orthogonal and diagonal influence.
        for (y = 1; y < HEIGHT-1; y++) {
            orthogonal = present[x-1][y  ] + present[x  ][y-1] + present[x  ][y+1] + present[x+1][y  ];
            diagonal   = present[x-1][y-1] + present[x-1][y+1] + present[x+1][y-1] + present[x+1][y+1];
            trend[x][y] = .5 * orthogonal + .25 * diagonal - present[x][y] - past[x][y]; // wave speed = 1
    }}

In 2-D, the trend extrapolation may be simplified to:
    trend[x][y] = .5 * orthogonal - past[x][y];
    Please note that .5 * orthogonal implies the relative influence of only 2 cells. Consequently, it is
    easier to balance to unity: 2 - 1 = 1). This produces a slower wave speed: 0,7071 pixel per loop (1 / sqrt(2)).
    However, it is still possible to add the present[x][y] variable (positive) in order to slow
    down the wave speed (for example to produce lens effects). One must modify "waveSpeed * present[x][y]"
    and balance to unity with the rest of the influence so that any slower variable speeds are possible.

    ################################################################################################

    JOCELYN MARCOTTE'S 3-D WAVE ALGORITHM (created: Jan. 2006) #####################################

    ################################################################################################
*/
    past = present;  // line 1 of algorithm.
    present = trend; // line 2 of algorithm. Using pointers in a rotation this way is very, very fast (implemented by Mr. Marcotte himself).
    trend = past;    // there are only 2 arrays of variables available in RAM while the algorithm apparently uses 3!

    for (x = 1; x < WIDTH-1; x++) {
        for (y = 1; y < HEIGHT-1; y++) { // summation of "present" amplitude values in a 3 x 3 x 3 = 27 pixels cubic volume.
            for (z = 1; z < DEPTH-1; z++) {
                orthogonal = present[x-1][y  ][z  ] + present[x  ][y-1][z  ] + present[x  ][y+1][z  ]
                           + present[x  ][y  ][z-1] + present[x  ][y  ][z+1] + present[x+1][y  ][z  ];       // 6 orthogonal cells, gain = 3 / 13
                diagonal   = present[x-1][y-1][z  ] + present[x  ][y-1][z-1] + present[x+1][y-1][z  ]
                           + present[x-1][y  ][z-1] + present[x  ][y-1][z+1] + present[x+1][y  ][z-1]
                           + present[x-1][y  ][z+1] + present[x  ][y+1][z-1] + present[x+1][y  ][z+1]
                           + present[x-1][y+1][z  ] + present[x  ][y+1][z+1] + present[x+1][y+1][z  ];        // 12 diagonal cells, gain = 3 / 26
                vertices   = present[x-1][y-1][z-1] + present[x-1][y-1][z+1] + present[x-1][y+1][z-1] + present[x-1][y+1][z+1]
                           + present[x+1][y-1][z-1] + present[x+1][y-1][z+1] + present[x+1][y+1][z-1] + present[x+1][y+1][z+1]; // 8, gain 1 / 13
                trend[x][y][z] = .23076923 * orthogonal + .1153846 * diagonal + .076923 * vertices - 1.384615 * present[x][y][z] - past[x][y][z];
    }}}
}

